/*
 * CollisionHandlerPhysicsNode.h
 *
 * Created 8/9/2009 By Johnny Huynh
 *
 * Version 00.00.02 8/22/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef COLLISION_HANDLER_PHYSICS_NODE_H
 #define COLLISION_HANDLER_PHYSICS_NODE_H
 
 /**
  * CAVEATS:
  * 1)  If any combination of CollisionHandlerPhysicsNode and other types of 
  *     CollisionHandlerEventNode are to be used for an Object, the Object
  *     must be converted into an ActorNode before initializing any of the 
  *     CollisionhandlerEventNode (or the CollisionHandlerPhysicsNode 
  *     must be initialized first); use the static function, 
  *     CollisionHandlerPhysicsNode<T>::convert_to_actor_node( NodePath& ),
  *     on the Object.
  * 2)  When CollisionHandlerPhysicsNode is initialized or attached to a NodePath,
  *     the NodePath will automatically lose its current physics (PhysicsObject)
  *     properties and receive this CollisionHandlerPhysicsNode's physics 
  *     (PhysicsObject) properties.
  */
 
 template <typename T> class CollisionHandlerPhysicsNode;
 
 #include "nodePath.h"
 
 // For collision handling
 #include "CollisionHandlerEventNode.h"
 #include "actorNode.h"
 #include "physicsCollisionHandler.h"
 
 #include "global.h"
 #include "ObjectCollection.h"
 
 /**
  * Class specification for CollisionHandlerPhysicsNode
  */
 template <typename T>
 class CollisionHandlerPhysicsNode : public CollisionHandlerEventNode<T>
 {
 // Data Members
 private:
    PT(ActorNode) _actor_node_Ptr;
    
 // Special Purpose Functions
 protected:
    CollisionHandlerPhysicsNode( const CollisionHandlerPhysicsNode<T>& chpn, PhysicsCollisionHandler* collision_handler_Ptr ); // for get_copy() function
 
 // Static Functions
 public:
    static inline void convert_to_actor_node( NodePath& node_path );
 
 // Local Functions
 public:
    CollisionHandlerPhysicsNode( const std::string& name );
    CollisionHandlerPhysicsNode( const CollisionHandlerPhysicsNode<T>& chpn );
    virtual ~CollisionHandlerPhysicsNode();
    //virtual inline CollisionHandlerPhysicsNode<T>& operator=( const CollisionHandlerPhysicsNode<T>& chpn );
    virtual inline PT(CollisionHandlerEventNode<T>) get_copy() const;
    //virtual inline void initialize_collision_handler( NodePath& node_path );
    virtual inline void reparent_to( NodePath& node_path ); // can be treated like a NodePath function
    //virtual inline void detach_node(); // can be treated like a NodePath function
    
    // NodePath functions
    /*virtual inline void remove_node() 
    { 
        if ( _collision_handler_Ptr != NULL )
        {
            PT(PhysicsCollisionHandler) physics_handler_Ptr( DCAST(PhysicsCollisionHandler, _collision_handler_Ptr) );
            physics_handler_Ptr->clear_center();
            physics_handler_Ptr->clear_colliders();
        }
          
        CollisionHandlerEventNode<T>::remove_node();
    }*/
    
 // Private Functions
 private:
    virtual inline void initialize_collision_handler();
    
 // Friend Functions
 public:
 
 };
 
 /** STATIC FUNCTIONS **/
 
 /**
  * convert_to_actor_node() converts the specified NodePath into an ActorNode if
  * it is not already an ActorNode.
  *
  * @param (NodePath&) node_path
  */
 template <typename T>
 inline void CollisionHandlerPhysicsNode<T>::convert_to_actor_node( NodePath& node_path )
 {
    // Rearrange attachment of PandaNodes
    // Top: actor_node_path
    // 2nd: node_path
    PT(ActorNode) actor_node_Ptr( new ActorNode( node_path.get_name() ) );
    NodePath actor_node_path( node_path.get_parent().attach_new_node( actor_node_Ptr ) );
    actor_node_path.set_name( node_path.get_name() );//"Physics Actor Node Path of " + node_path.get_name() );
    actor_node_path.set_tag( "isActorNode", "true" );
    actor_node_path.set_pos( node_path.get_pos() );
    actor_node_path.set_hpr( node_path.get_hpr() );
    node_path.set_pos( ZERO, ZERO, ZERO );
    node_path.set_hpr( ZERO, ZERO, ZERO );
    
    node_path.reparent_to( actor_node_path );
    
    // change the node_path to point to actor_node_path for the programmer to control 
    // actor_node_path as the object
    // the pre-loaded animations should still be binded with the 2nd node_path
    // the animations should still be controllable because of the AnimControl
    node_path = actor_node_path;
    
    //printf("NodePathKey: %1d\n", node_path.get_key() );
    //printf("ActorNodePathKey: %1d\n", actor_node_path.get_key() );
 }
 
 /** PROTECTED FUNCTIONS **/
 
 /**
  * Special Purpose Copy Constructor
  */
 template <typename T>
 CollisionHandlerPhysicsNode<T>::CollisionHandlerPhysicsNode( const CollisionHandlerPhysicsNode<T>& chpn, PhysicsCollisionHandler* collision_handler_Ptr )
                               : CollisionHandlerEventNode( chpn, collision_handler_Ptr ),
                                 _actor_node_Ptr( new ActorNode( chpn.get_name() + " Actor Node" ) )
 {
    
 }
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  *
  * @param (const std::string&) name - name for this CollisionHandlerPhysicsNode
  */
 template <typename T>
 CollisionHandlerPhysicsNode<T>::CollisionHandlerPhysicsNode( const std::string& name )
                               : CollisionHandlerEventNode<T>( name, true, false ),
                                 _actor_node_Ptr( new ActorNode( name + " Actor Node" ) )
 {
    nassertv( !_actor_node_Ptr.is_null() );
    CollisionHandlerPhysicsNode<T>::initialize_collision_handler();
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 CollisionHandlerPhysicsNode<T>::CollisionHandlerPhysicsNode( const CollisionHandlerPhysicsNode<T>& chpn )
                                : CollisionHandlerEventNode<T>( chpn ),
                                  _actor_node_Ptr( chpn._actor_node_Ptr )
 {
 
 }
 
 /**
  * Destructor
  */
 template <typename T>
 CollisionHandlerPhysicsNode<T>::~CollisionHandlerPhysicsNode()
 {
    if ( _collision_handler_Ptr != NULL )
    {
        PT(PhysicsCollisionHandler) physics_handler_Ptr( DCAST(PhysicsCollisionHandler, _collision_handler_Ptr) );
        physics_handler_Ptr->clear_center();
        physics_handler_Ptr->clear_colliders();
    }
 }
 
 /**
  * operator=() copies the content of the specified CollisionHandlerPhysicsNode
  * to this CollisionHandlerPhysicsNode.
  *
  * @param (const CollisionHandlerPhysicsNode<T>& chpn )
  * @return CollisionHandlerPhysicsNode<T>&
  */
 /*template <typename T>
 inline CollisionHandlerPhysicsNode<T>& CollisionHandlerPhysicsNode<T>::operator=( const CollisionHandlerPhysicsNode<T>& chpn )
 {
    CollisionHandlerNode<T>::operator=( chpn );
    
    return *this;
 }*/
 
 /**
  * get_copy() returns a pointer to a CollisionHandlerEventNode that hard copies this
  * CollisionHandlerPhysicsNode.
  *
  * @return PT(CollisionHandlerEventNode<T>)
  */
 template <typename T>
 inline PT(CollisionHandlerEventNode<T>) CollisionHandlerPhysicsNode<T>::get_copy() const
 {
    return reinterpret_cast<CollisionHandlerEventNode<T>*>(new CollisionHandlerPhysicsNode<T>( *this, new PhysicsCollisionHandler() ));
 }
 
 /**
  * reparent_to() attaches this collision handler node to the specified NodePath.
  */
 template <typename T>
 inline void CollisionHandlerPhysicsNode<T>::reparent_to( NodePath& node_path )
 {
    // if the collision handler has not been initialized yet
    //if ( _collision_handler_Ptr == NULL )
        //initialize_collision_handler();
    
    // if the node_path is not an ActorNode,
    // convert the node_path into an ActorNode
    if ( node_path.get_tag( "isActorNode" ) != "true" )
    {
        // Rearrange attachment of PandaNodes
        // Top: actor_node_path
        // 2nd: node_path, *this
        CollisionHandlerPhysicsNode<T>::convert_to_actor_node( node_path );
    }
    
    // assert the node_path is an actor node
    //nassertv( node_path.get_tag( "isActorNode" ) == "true" );
    
    // set the physics properties of the node_path to have the physics properties of this collider
    PT(ActorNode) np_actor_node_Ptr( DCAST(ActorNode, node_path.node()) );
    //np_actor_node_Ptr->add_physicals_from( *_actor_node_Ptr );
    *np_actor_node_Ptr->get_physics_object() = *_actor_node_Ptr->get_physics_object();
    
    PT(PhysicsCollisionHandler) physics_handler_Ptr( DCAST(PhysicsCollisionHandler, _collision_handler_Ptr) );
    physics_handler_Ptr->remove_collider( *this ); // remove the previous entry for the physics handler
    // add_collider( collision_solid_node_path, node_path_effected_by_collision )
    physics_handler_Ptr->add_collider( *this, node_path ); // add the new entry for the physics handler
    
    CollisionHandlerEventNode<T>::reparent_to( node_path );
 }
 
 /**
  * detach_node() makes collision impossible for this node by detaching this collision handler.
  */
 /*template <typename T>
 inline void CollisionHandlerPhysicsNode<T>::detach_node()
 {
    NodePath node_path( _collider.get_parent() );
    
    // if this collider has a parent
    if ( !node_path.is_empty() )
    {
        // assert the node_path is an actor node
        nassertv( node_path.get_tag( "isActorNode" ) == "true" );
        
        PT(ActorNode) np_actor_node_Ptr( DCAST(ActorNode, node_path.node()) );
        
        // Remove all physicals of this collider from the node_path
        for ( size_t i( 0 ); i < _actor_node_Ptr->get_num_physicals(); ++i )
        {
            np_actor_node_Ptr->remove_physical( _actor_node_Ptr->get_physical( i ) );
        }
        
        CollisionHandlerEventNode<T>::detach_node();
    }
 }*/
 
 /** PRIVATE FUNCTIONS **/
 
 /**
  * initialize_collision_handler() initializes the collision handling for this CollisionHandlerEventNode.
  * A CollisionHandlerPhysicsNode is initialized to be an "into" and "from" collider.
  */
 template <typename T>
 inline void CollisionHandlerPhysicsNode<T>::initialize_collision_handler()
 {
    PT(PhysicsCollisionHandler) physics_handler_Ptr( new PhysicsCollisionHandler() );
    CollisionHandlerEventNode<T>::_collision_handler_Ptr = physics_handler_Ptr;
    
    // assert the collision handler has been allocated
    nassertv( !physics_handler_Ptr.is_null() );
    
    // set the physics properties of the node_path to have the physics properties of this collider
    //PT(ActorNode) np_actor_node_Ptr( DCAST(ActorNode, node_path.node()) );
    //np_actor_node_Ptr->add_physicals_from( *_actor_node_Ptr );
    //*np_actor_node_Ptr->get_physics_object() = *_actor_node_Ptr->get_physics_object();
    
    // add_collider( collision_solid_node_path, node_path_effected_by_collision )
    //physics_handler_Ptr->add_collider( *this, parent );
    //physics_handler_Ptr->add_collider( _collider, actor_node_path );
    
    //CollisionHandlerEventNode<T>::turn_on_from_collision();
    //_is_from_collider = true;
 }
 
 /** FRIEND FUNCTIONS **/
 
 #endif // COLLISION_HANDLER_PHYSICS_NODE_H